--# Main
require(asset.documents.Craft.Touches)
require(asset.documents.Craft.Cameras)

local useEarth = false
local photoEarth = false
local pendingCubemapAsset = nil
local showImportOverlay = false

function setup()
    print("NOTE: project requires importing assets realisticEarth and Earth_cube_map to show Earth textures")
    setupGlobals()
    setupCraftScene()
    
    cubeMapSphere = CubeMapSphere(scene, PLANET_RADIUS, 512)
    cubeMapSphere:setDefaultFaces()          -- build quick labeled faces
    cubeMapSphere:bakeFacesToRT()            -- <-- render the 6 images into the RT cubemap
    cubeMapSphere:usePatchMaterial()         -- <-- put the stretch shader on the sphere
end

function setupGlobals()
    viewer.mode = FULLSCREEN
    PLANET_RADIUS = 200
    parameter.boolean("ShowCubeMap", true)
    parameter.boolean("UseEarthTextures", false, function(v)
        useEarth = v
        if not cubeMapSphere then return end
        
        if useEarth then
            pendingCubemapAsset = photoEarth and asset.realisticEarth or asset.Earth_cube_map
            showImportOverlay = false
        else
            cubeMapSphere:setDefaultFaces()
            cubeMapSphere:bakeFacesToRT()    -- update RT from the new faces
            cubeMapSphere:usePatchMaterial()
            pendingCubemapAsset = nil
            showImportOverlay = false
        end
    end)
    
    parameter.boolean("PhotorealisticEarth", false, function(v)
        photoEarth = v
        if not cubeMapSphere then return end
        if useEarth then
            pendingCubemapAsset = photoEarth and asset.realisticEarth or asset.Earth_cube_map
            showImportOverlay = false
        end
    end)
    parameter.boolean("DrawingMode", false)
    
    parameter.number("Brush Size", 5, 50, 20, function(v)
        if cubeMapSphere then cubeMapSphere:setBrushSize(v) end
    end)
end

function setupCraftScene()
    scene = craft.scene()
    scene.ambientColor = color(180, 180, 200)
    scene.camera.position = vec3(0, 0, 800)
    local cam = scene.camera:get(craft.camera)
    cam.nearPlane, cam.farPlane = 10, 4000
    vwr = scene.camera:add(OrbitViewer, vec3(0, -200, 0), 1400, 300, 2400)
    
    local sky = scene.sky.material
    sky.sky     = color(30, 121, 140)
    sky.horizon = color(31, 44, 73)
    sky.ground  = color(48, 31, 71)
end

-- one-frame overlay then import heavy image next frame
function processCubemapLoading()
    if not cubeMapSphere or pendingCubemapAsset == nil then return end
    if not showImportOverlay then
        pushStyle() fill(255) fontSize(28) textAlign(CENTER)
        text("Importing cubemap…", WIDTH/2, HEIGHT/2)
        popStyle()
        showImportOverlay = true
        return
    else
        cubeMapSphere:loadCubemapImage(pendingCubemapAsset) -- fills self.cubeImages
        cubeMapSphere:bakeFacesToRT()                       -- push into RT
        cubeMapSphere:usePatchMaterial()
        pendingCubemapAsset = nil
        showImportOverlay = false
    end
end

function draw()
    scene:update(DeltaTime)
    scene:draw()
    
    processCubemapLoading()
    
    if ShowCubeMap and cubeMapSphere then
        cubeMapSphere:drawPreview()
    end
end

function touched(t)
    if DrawingMode and cubeMapSphere then
        local handled = cubeMapSphere:handleTouch(t)
        if not handled then vwr:touched(t) end
    else
        vwr:touched(t)
    end
end

function copyImageFromDocumentsToProject(n)
    local img = readImage(asset.documents .. n)
    saveImage(asset .. n, img)
end