-- Bottom alignment demo (Codea)
-- Shows why text() y doesn't behave like a consistent baseline,
-- and demonstrates a pixel-based workaround.

viewer.mode = FULLSCREEN

function draw()
    background(245, 245, 245)   
    drawAlignedBottomsUsingRawYValues(DEMO.str1, DEMO.fs1, DEMO.str2, DEMO.fs2)
    drawAlignedBottomsUsingTextSize(DEMO.str1, DEMO.fs1, DEMO.str2, DEMO.fs2)
    drawAlignedBottomsUsingPixelDetection(DEMO.str1, DEMO.fs1, DEMO.str2, DEMO.fs2)
    
    drawExplanations()
end

function touched(t)
    if t.state ~= ENDED then return end
    
    local cases = {
        { "abc:", 54, "abc", 120 },
        { "hq:", 80, "hq", 120 },
        { "Score:", 54, "91", 120 },
    }
    
    DEMO._i = (DEMO._i or 0) + 1
    local c = cases[((DEMO._i - 1) % #cases) + 1]
    DEMO.str1, DEMO.fs1, DEMO.str2, DEMO.fs2 = c[1], c[2], c[3], c[4]
end

------------------------------------------------------------
-- 1) Raw Y values (same y for both draws)
------------------------------------------------------------
function drawAlignedBottomsUsingRawYValues(string1, fontSize1, string2, fontSize2)
    local leftX   = 60
    local bottomY = HEIGHT * 0.72
    
    -- guide line (where we'd like the bottoms to align)
    pushStyle()
    stroke(255, 0, 120, 180)
    strokeWidth(2)
    line(40, bottomY, WIDTH - 40, bottomY)
    popStyle()
    
    pushStyle()
    textMode(CORNER)
    textAlign(LEFT)
    fill(20, 40, 60, 255)
    
    fontSize(fontSize1)
    text(string1, leftX, bottomY)
    
    local w1 = textSize(string1)
    fontSize(fontSize2)
    text(string2, leftX + w1 + 16, bottomY)
    
    popStyle()
end

------------------------------------------------------------
-- 2) textSize-based attempt:
--    Treat height as if it lets us compute a bottom-aligned y.
--    (This is intentionally "what you'd try" and it demonstrates
--     why it still doesn't align bottoms in Codea.)
------------------------------------------------------------
function drawAlignedBottomsUsingTextSize(string1, fontSize1, string2, fontSize2)
    local leftX   = 60
    local bottomY = HEIGHT * 0.50
    
    pushStyle()
    stroke(255, 0, 120, 180)
    strokeWidth(2)
    line(40, bottomY, WIDTH - 40, bottomY)
    popStyle()
    
    pushStyle()
    textMode(CORNER)
    textAlign(LEFT)
    fill(20, 40, 60, 255)
    
    fontSize(fontSize1)
    local w1, h1 = textSize(string1)
    local x1 = leftX
    local y1 = bottomY       -- bottom of the *bounding box* sits on the red line
    text(string1, x1, y1)
    
    fontSize(fontSize2)
    local w2, h2 = textSize(string2)
    local x2 = leftX + w1 + 16
    local y2 = bottomY
    text(string2, x2, y2)
    
    -- thin purple bounding boxes from textSize() (bottom-aligned to the red line)
    noFill()
    stroke(150, 0, 220, 180)
    strokeWidth(3)
    rectMode(CORNER)
    rect(x1, y1, w1, h1)
    rect(x2, y2, w2, h2)
    
    popStyle()
end

------------------------------------------------------------
-- 3) Pixel detection (true visual-bottom alignment)
------------------------------------------------------------
function drawAlignedBottomsUsingPixelDetection(string1, fontSize1, string2, fontSize2)
    local leftX   = 60
    local bottomY = HEIGHT * 0.28
    
    pushStyle()
    stroke(255, 0, 120, 180)
    strokeWidth(2)
    line(40, bottomY, WIDTH - 40, bottomY)
    popStyle()
    
    pushStyle()
    fontSize(fontSize1)
    local w1 = textSize(string1)
    fill(0)
    drawTextIgnoringDescendersWithBottomAt(string1, leftX, bottomY)
    fontSize(fontSize2)
    drawTextIgnoringDescendersWithBottomAt(string2, leftX + w1 + 16, bottomY)
    
    popStyle()    
end

function drawExplanations()
    -- labels under the red guide lines + footer hint
    pushStyle()
    fill(60, 60, 60, 255)
    fontSize(18)
    textMode(CORNER)
    textAlign(LEFT)
    
    local yRaw = HEIGHT * 0.72
    local yTS  = HEIGHT * 0.50
    local yPix = HEIGHT * 0.28
    local labelDy = 26
    
    text("Raw Y (just using same y for both)",         60, yRaw - labelDy)
    text("Using textSize() bounds (purple boxes)",     60, yTS  - labelDy)
    text("Pixel detection (visual-bottom aligned)",    60, yPix - labelDy)
    
    fill(80, 80, 80, 255)
    fontSize(16)
    textMode(CENTER)
    textAlign(CENTER)
    text("Tap screen to cycle test strings", WIDTH/2, 94)
    
    popStyle()
end

DEMO = {
    str1 = "Score:",
    fs1  = 54,
    str2 = "91",
    fs2  = 120,
}

